<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PracticaProfesional;
use App\Models\Postulante;
use App\Models\ModuloFormativo;
use App\Models\Docente;
use App\Models\Matricula;
use Illuminate\Support\Facades\File;

class PracticaProfesionalController extends Controller
{
    public function index()
    {
        $practicas = PracticaProfesional::with(['postulante', 'modulo', 'asesor'])->get();
        $postulantes = Postulante::all();
        $modulos = ModuloFormativo::all();
        $asesores = Docente::all();
        return view('practicass.index', compact('practicas','postulantes', 'modulos', 'asesores'));
    }

    public function create()
    {
        $postulantes = Postulante::all();
        $modulos = ModuloFormativo::all();
        $asesores = Docente::all();

        return view('practicass.create', compact('postulantes', 'modulos', 'asesores'));
    }

    public function store(Request $request)
{
    try {
        // Validar los datos
        $request->validate([
            'postulante_id' => 'required|exists:postulantes,id_postulante',
            'id_modulo' => 'required|exists:modulos_formativos,id_modulo',
            'id_asesor' => 'required|exists:docentes,id',
            'empresa_nombre' => 'required|string|max:255',
            'empresa_ruc' => 'required|string|max:20',
            'fecha_inicio' => 'required|date',
            'fecha_fin' => 'required|date|after_or_equal:fecha_inicio',
            'fecha_registro' => 'required|date',
            'nota' => 'nullable|numeric',
            'estado' => 'nullable|string|in:pendiente,aprobado,desaprobado',
            'archivo_evidencia' => 'nullable|file|mimes:pdf,doc,docx,png,jpg,jpeg',
            'observacion' => 'nullable|string',
        ]);

        // Verificar si ya existe una práctica con el mismo postulante y módulo
        $existe = PracticaProfesional::where('postulante_id', $request->postulante_id)
            ->where('id_modulo', $request->id_modulo)
            ->exists();

        if ($existe) {
            return response()->json([
                'success' => false,
                'message' => 'Ya existe un registro de práctica para este estudiante y módulo.'
            ], 422);
        }

        // Manejo del archivo
        $rutaArchivo = null;
        if ($request->hasFile('archivo_evidencia')) {
            $archivo = $request->file('archivo_evidencia');
            $nombreArchivo = time().'_'.$archivo->getClientOriginalName();

            $rutaDestino = public_path('practicas');
            if (!File::exists($rutaDestino)) {
                File::makeDirectory($rutaDestino, 0755, true);
            }

            $archivo->move($rutaDestino, $nombreArchivo);
            $rutaArchivo = 'practicas/' . $nombreArchivo;
        }

        // Crear la práctica
        PracticaProfesional::create([
            'postulante_id' => $request->postulante_id,
            'id_modulo' => $request->id_modulo,
            'id_asesor' => $request->id_asesor,
            'empresa_nombre' => $request->empresa_nombre,
            'empresa_ruc' => $request->empresa_ruc,
            'fecha_inicio' => $request->fecha_inicio,
            'fecha_fin' => $request->fecha_fin,
            'fecha_registro' => $request->fecha_registro,
            'nota' => $request->nota,
            'estado' => $request->estado,
            'archivo_evidencia' => $rutaArchivo,
            'observacion' => $request->observacion,
        ]);

        return response()->json(['success' => true, 'message' => 'Práctica registrada correctamente.']);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error al registrar la práctica.',
            'error' => $e->getMessage()
        ], 500);
    }
}

    public function show($id)
    {
        $practica = PracticaProfesional::with(['postulante', 'modulo', 'asesor'])->findOrFail($id);
        return view('practicass.show', compact('practica'));
    }

    public function edit($id)
{
    $practica = PracticaProfesional::with(['asesor', 'modulo', 'postulante'])->findOrFail($id);

    return response()->json($practica);
}


    public function update(Request $request, $id)
{
    $practica = PracticaProfesional::findOrFail($id);

    $request->validate([
        'id_asesor' => 'required|exists:docentes,id',
        'empresa_nombre' => 'required|string|max:150',
        'empresa_ruc' => 'required|string|max:20',
        'fecha_inicio' => 'required|date',
        'fecha_fin' => 'required|date|after_or_equal:fecha_inicio',
        'nota' => 'nullable|numeric|between:0,20',
        'observacion' => 'nullable|string',
        'archivo_evidencia' => 'nullable|file|mimes:pdf,doc,docx,jpg,png|max:2048',
        'estado' => 'in:pendiente,aprobado,desaprobado'
    ]);

    $data = $request->except('archivo_evidencia');

    if ($request->hasFile('archivo_evidencia')) {
        if ($practica->archivo_evidencia && File::exists(public_path($practica->archivo_evidencia))) {
            File::delete(public_path($practica->archivo_evidencia));
        }

        $archivo = $request->file('archivo_evidencia');
        $nombreArchivo = time() . '_' . $archivo->getClientOriginalName();
        $ruta = public_path('practicas');

        if (!File::exists($ruta)) {
            File::makeDirectory($ruta, 0755, true);
        }

        $archivo->move($ruta, $nombreArchivo);
        $data['archivo_evidencia'] = 'practicas/' . $nombreArchivo;
    }

    $practica->update($data);

    // ✅ Verifica si la petición es AJAX
    if ($request->ajax()) {
        return response()->json([
            'success' => true,
            'message' => 'Práctica actualizada correctamente.'
        ]);
    }

    // 🧭 Si no es AJAX, redirecciona como siempre
    return redirect()->route('practicass.index')->with('success', 'Práctica actualizada correctamente.');
}


   public function destroy($id)
{
    $practica = PracticaProfesional::findOrFail($id);

    // Eliminar archivo si existe
    if ($practica->archivo_evidencia && File::exists(public_path($practica->archivo_evidencia))) {
        File::delete(public_path($practica->archivo_evidencia));
    }

    $practica->delete();

    if (request()->ajax()) {
        return response()->json([
            'success' => true,
            'message' => 'Práctica eliminada correctamente.'
        ]);
    }

    return redirect()->route('practicass.index')->with('success', 'Práctica eliminada correctamente.');
}


    public function modulosPorPostulante($id)
{
    // Busca la última matrícula del postulante
    $matricula = Matricula::where('postulante_id', $id)->orderByDesc('created_at')->first();

    // Verifica si existe la matrícula y si tiene plan de estudio
    if (!$matricula || !$matricula->plan_estudio_id) {
        return response()->json([
            'success' => false,
            'message' => 'No se encontró una matrícula válida con plan de estudio para este postulante.',
            'modulos' => []
        ]);
    }

    // Obtiene módulos del plan de estudio asociado
    $modulos = ModuloFormativo::where('id_plan_estudio', $matricula->plan_estudio_id)
        ->select('id_modulo', 'nombre')
        ->get();

    return response()->json([
        'success' => true,
        'modulos' => $modulos
    ]);
}

}
